<?php
/**
 * 去重操作API
 */

require_once dirname(__FILE__) . '/../config.php';
require_once dirname(__FILE__) . '/../class/QuchongDuplication.php';

// 防止超时
set_time_limit(0);
ini_set('memory_limit', '512M');

// 关闭错误显示，防止破坏JSON格式
ini_set('display_errors', 0);
error_reporting(E_ALL);

// 注册错误处理函数
register_shutdown_function(function() {
    $error = error_get_last();
    if ($error && ($error['type'] === E_ERROR || $error['type'] === E_PARSE || $error['type'] === E_CORE_ERROR || $error['type'] === E_COMPILE_ERROR)) {
        header('Content-Type: application/json; charset=utf-8');
        echo json_encode(['success' => false, 'message' => '系统错误: ' . $error['message'] . ' in ' . $error['file'] . ':' . $error['line']]);
    }
});

header('Content-Type: application/json; charset=utf-8');

// 支持 GET 和 POST 的 action 参数
$action = $_GET['action'] ?? $_POST['action'] ?? '';

// 初始化日志
@mkdir(dirname(__FILE__) . '/../logs', 0777, true);

try {
    // 验证PDO连接
    if (!isset($pdo) || !$pdo instanceof PDO) {
        throw new Exception('数据库连接失败');
    }
    
    $quchong = new QuchongDuplication($pdo, DB_PREFIX);
    $response = ['success' => false, 'message' => '未知操作'];

    switch ($action) {
        case 'detect':
            // 检测重复
            $mode = $_POST['mode'] ?? 'title';
            $limit = intval($_POST['limit'] ?? 1000);
            
            QuchongDuplication::log("开始检测重复: mode=$mode, limit=$limit");
            $result = $quchong->detectDuplicate($mode, $limit);
            
            if (isset($result['error'])) {
                $response = ['success' => false, 'message' => $result['error']];
                QuchongDuplication::log("检测失败: " . $result['error']);
            } else {
                $response = ['success' => true, 'data' => $result];
                QuchongDuplication::log("检测完成: 找到" . count($result) . "组重复");
            }
            break;

        case 'remove':
            // 移除重复
            $ids = $_POST['ids'] ?? [];
            $rule = $_POST['rule'] ?? 'newest';
            $keepId = $_POST['keep_id'] ?? null;
            
            if (is_string($ids)) {
                $ids = array_map('intval', explode(',', $ids));
            }
            
            $result = $quchong->removeDuplicate($ids, $rule, $keepId);
            $response = $result;
            
            if ($result['success']) {
                QuchongDuplication::log("执行去重操作: 模式=$rule, 删除视频数=" . $result['deleted_count']);
            }
            break;

        case 'merge':
            // 合并视频
            $keepId = intval($_POST['keep_id'] ?? 0);
            $deleteIds = $_POST['delete_ids'] ?? [];
            
            if (is_string($deleteIds)) {
                $deleteIds = array_map('intval', explode(',', $deleteIds));
            }
            
            if (!$keepId || empty($deleteIds)) {
                $response = ['success' => false, 'message' => '参数不完整'];
                break;
            }
            
            $result = $quchong->removeDuplicate(array_merge([$keepId], $deleteIds), 'merge', $keepId);
            $response = $result;
            
            if ($result['success']) {
                QuchongDuplication::log("执行合并操作: 保留视频ID=$keepId, 合并数=" . $result['merged_count']);
            }
            break;

        case 'stats':
            // 获取统计
            QuchongDuplication::log("获取统计信息");
            $stats = $quchong->getStats();
            
            if (isset($stats['error'])) {
                $response = ['success' => false, 'message' => $stats['error']];
                QuchongDuplication::log("统计失败: " . $stats['error']);
            } else {
                $response = ['success' => true, 'data' => $stats];
            }
            break;

        case 'modes':
            // 获取所有模式
            $response = ['success' => true, 'data' => QuchongDuplication::getModes()];
            break;

        case 'rules':
            // 获取所有规则
            $response = ['success' => true, 'data' => QuchongDuplication::getRules()];
            break;

        default:
            $response = ['success' => false, 'message' => '不支持的操作: ' . $action];
    }
} catch (Exception $e) {
    $response = ['success' => false, 'message' => $e->getMessage()];
    QuchongDuplication::log("API错误: " . $e->getMessage());
}

echo json_encode($response, JSON_UNESCAPED_UNICODE);
?>

