<?php
/**
 * 宝塔定时任务脚本 - 自动视频去重
 * 
 * 使用方法：
 * 1. 将此文件上传到服务器
 * 2. 在宝塔计划任务中添加：Shell脚本
 * 3. 脚本内容：/usr/bin/php /www/wwwroot/你的网站目录/quchong/api/cron.php
 * 
 * 配置说明：
 * - 默认去重模式：按标题+分类组合 (title_type)
 * - 默认保留规则：合并视频（替换播放源） (merge)
 * - 每次处理数量：1000条
 */

// 设置脚本不超时
set_time_limit(0);
ini_set('memory_limit', '1024M');

// 引入核心文件
require_once dirname(__FILE__) . '/../config.php';
require_once dirname(__FILE__) . '/../class/QuchongDuplication.php';

// 检查是否在命令行模式下运行
if (php_sapi_name() !== 'cli' && !isset($_GET['key'])) {
    die('此脚本仅限命令行运行，或需要安全密钥');
}

// 定义要执行的任务列表
$tasks = [
    [
        'mode' => 'title_type', // 任务1：按标题+分类组合
        'rule' => 'merge'       // 合并视频
    ],
    [
        'mode' => 'url',        // 任务2：按播放链接
        'rule' => 'merge'       // 合并视频
    ]
];

$LIMIT = 1000; // 每次查询数量

try {
    // 1. 初始化
    if (!isset($pdo) || !$pdo instanceof PDO) {
        throw new Exception('Database connection failed');
    }
    
    $quchong = new QuchongDuplication($pdo, DB_PREFIX);

    // 循环执行任务
    foreach ($tasks as $i => $task) {
        $MODE = $task['mode'];
        $RULE = $task['rule'];

        echo "\n================================\n";
        echo "Executing Task " . ($i + 1) . "/" . count($tasks) . "\n";
        echo "Mode: $MODE\n";
        echo "Rule: $RULE\n";
        echo "Limit: $LIMIT\n";
        echo "--------------------------------\n";

        // 2. 检测重复
        echo "Detecting duplicates...\n";
        $results = $quchong->detectDuplicate($MODE, $LIMIT);
        
        if (isset($results['error'])) {
            echo "Error detecting: " . $results['error'] . "\n";
            QuchongDuplication::log("Cron task error [$MODE]: " . $results['error']);
            continue; // 跳过当前任务，继续下一个
        }
        
        $totalGroups = count($results);
        echo "Found $totalGroups duplicate groups.\n";
        
        if ($totalGroups === 0) {
            echo "No duplicates found for this mode.\n";
            continue;
        }
        
        // 3. 循环处理
        $successCount = 0;
        $failCount = 0;
        
        foreach ($results as $index => $group) {
            $ids = explode(',', $group['ids']);
            $ids = array_map('intval', $ids);
            
            $key = $group['vod_name'] ?? $group['key'] ?? $group['vod_play_url'] ?? 'Unknown';
            // 截取过长的key显示
            if (strlen($key) > 50) $key = substr($key, 0, 47) . '...';
            
            echo "[" . ($index + 1) . "/$totalGroups] Processing: $key (IDs: " . count($ids) . ")... ";
            
            // 调用去重逻辑
            $result = $quchong->removeDuplicate($ids, $RULE);
            
            if ($result['success']) {
                echo "SUCCESS\n";
                $successCount++;
            } else {
                echo "FAILED: " . $result['message'] . "\n";
                $failCount++;
            }
        }
        
        echo "--------------------------------\n";
        echo "Task " . ($i + 1) . " Completed.\n";
        echo "Success: $successCount\n";
        echo "Failed: $failCount\n";
        
        // 记录日志
        QuchongDuplication::log("Cron task finished. Mode: $MODE, Rule: $RULE, Total: $totalGroups, Success: $successCount");
    }

    echo "\n================================\n";
    echo "All tasks finished.\n";

} catch (Exception $e) {
    echo "Fatal Error: " . $e->getMessage() . "\n";
    QuchongDuplication::log("Cron task fatal error: " . $e->getMessage());
}
?>