<?php
/**
 * 视频去重核心类
 * @version 1.1.0
 */

class QuchongDuplication
{
    private $pdo;
    private $prefix;
    
    // 去重模式常量
    const MODE_TITLE = 'title';                    // 按标题完全匹配
    const MODE_TITLE_FUZZY = 'title_fuzzy';        // 按标题模糊匹配
    const MODE_URL = 'url';                        // 按播放链接
    const MODE_HASH = 'hash';                      // 按内容哈希值
    const MODE_TITLE_ACTOR = 'title_actor';        // 按标题+演员组合
    const MODE_TITLE_YEAR = 'title_year';          // 按标题+年代组合
    const MODE_TITLE_TYPE = 'title_type';          // 按标题+分类组合
    
    // 保留规则常量
    const RULE_NEWEST = 'newest';                  // 保留最新添加
    const RULE_OLDEST = 'oldest';                  // 保留最早添加
    const RULE_MOST_PLAYED = 'most_played';        // 保留播放最多
    const RULE_HIGHEST_SCORE = 'highest_score';    // 保留评分最高
    const RULE_MERGE = 'merge';                    // 合并视频（替换播放源）

    public function __construct($pdo, $prefix = 'mac_')
    {
        $this->pdo = $pdo;
        $this->prefix = $prefix;
    }

    /**
     * 获取所有去重模式
     */
    public static function getModes()
    {
        return [
            self::MODE_TITLE => '按标题完全匹配',
            self::MODE_TITLE_FUZZY => '按标题模糊匹配',
            self::MODE_URL => '按播放链接',
            self::MODE_HASH => '按内容哈希值',
            self::MODE_TITLE_ACTOR => '按标题+演员组合',
            self::MODE_TITLE_YEAR => '按标题+年代组合',
            self::MODE_TITLE_TYPE => '按标题+分类组合',
        ];
    }

    /**
     * 获取所有保留规则
     */
    public static function getRules()
    {
        return [
            self::RULE_NEWEST => '保留最新添加的视频',
            self::RULE_OLDEST => '保留最早添加的视频',
            self::RULE_MOST_PLAYED => '保留播放最多的视频',
            self::RULE_HIGHEST_SCORE => '保留评分最高的视频',
            self::RULE_MERGE => '合并视频（替换播放源）',
        ];
    }

    /**
     * 检测重复视频
     * @param string $mode 去重模式
     * @param int $limit 查询限制
     * @return array 重复视频组
     */
    public function detectDuplicate($mode = self::MODE_TITLE, $limit = 1000)
    {
        $sql = '';
        
        switch ($mode) {
            case self::MODE_TITLE:
                $sql = "SELECT `vod_name`, COUNT(*) as `count`, GROUP_CONCAT(`vod_id`) as `ids` 
                        FROM `{$this->prefix}vod` 
                        WHERE `vod_name` != '' 
                        GROUP BY `vod_name` 
                        HAVING `count` > 1 
                        LIMIT $limit";
                break;
                
            case self::MODE_TITLE_FUZZY:
                $sql = "SELECT SUBSTRING_INDEX(`vod_name`, ' ', 1) as `vod_name`, COUNT(*) as `count`, GROUP_CONCAT(`vod_id`) as `ids` 
                        FROM `{$this->prefix}vod` 
                        WHERE `vod_name` != '' 
                        GROUP BY SUBSTRING_INDEX(`vod_name`, ' ', 1) 
                        HAVING `count` > 1 
                        LIMIT $limit";
                break;
                
            case self::MODE_URL:
                $sql = "SELECT `vod_play_url`, COUNT(*) as `count`, GROUP_CONCAT(`vod_id`) as `ids` 
                        FROM `{$this->prefix}vod` 
                        WHERE `vod_play_url` != '' 
                        GROUP BY `vod_play_url` 
                        HAVING `count` > 1 
                        LIMIT $limit";
                break;
                
            case self::MODE_HASH:
                $sql = "SELECT MD5(CONCAT(`vod_name`, `vod_play_url`)) as `hash`, COUNT(*) as `count`, GROUP_CONCAT(`vod_id`) as `ids` 
                        FROM `{$this->prefix}vod` 
                        GROUP BY `hash` 
                        HAVING `count` > 1 
                        LIMIT $limit";
                break;
                
            case self::MODE_TITLE_ACTOR:
                $sql = "SELECT CONCAT(`vod_name`, '_', `vod_actor`) as `key`, COUNT(*) as `count`, GROUP_CONCAT(`vod_id`) as `ids` 
                        FROM `{$this->prefix}vod` 
                        WHERE `vod_name` != '' AND `vod_actor` != '' 
                        GROUP BY `key` 
                        HAVING `count` > 1 
                        LIMIT $limit";
                break;
                
            case self::MODE_TITLE_YEAR:
                // 使用 vod_year 字段，如果为空则尝试从 vod_time 获取
                $sql = "SELECT CONCAT(`vod_name`, '_', IF(`vod_year` != '' AND `vod_year` != '0', `vod_year`, FROM_UNIXTIME(`vod_time`, '%Y'))) as `key`, COUNT(*) as `count`, GROUP_CONCAT(`vod_id`) as `ids` 
                        FROM `{$this->prefix}vod` 
                        WHERE `vod_name` != '' 
                        GROUP BY `key` 
                        HAVING `count` > 1 
                        LIMIT $limit";
                break;
                
            case self::MODE_TITLE_TYPE:
                $sql = "SELECT CONCAT(`vod_name`, '_', `type_id`) as `key`, COUNT(*) as `count`, GROUP_CONCAT(`vod_id`) as `ids` 
                        FROM `{$this->prefix}vod` 
                        WHERE `vod_name` != '' AND `type_id` > 0 
                        GROUP BY `key` 
                        HAVING `count` > 1 
                        LIMIT $limit";
                break;
                
            default:
                return ['error' => '不支持的去重模式'];
        }
        
        try {
            // 增加GROUP_CONCAT长度限制，防止ID过多被截断
            $this->pdo->exec("SET SESSION group_concat_max_len = 1000000");
            
            $stmt = $this->pdo->query($sql);
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (Exception $e) {
            return ['error' => '查询失败: ' . $e->getMessage()];
        }
    }

    /**
     * 移除重复视频
     * @param array $ids 要删除的ID列表
     * @param string $rule 保留规则
     * @param int $keepId 要保留的ID（仅在合并时使用）
     * @return array 操作结果
     */
    public function removeDuplicate($ids, $rule = self::RULE_NEWEST, $keepId = null)
    {
        if (empty($ids) || !is_array($ids)) {
            return ['success' => false, 'message' => '参数错误'];
        }

        // 备份数据
        $this->backupVideos($ids);

        try {
            // 如果是合并模式且未指定保留ID，则自动选择最新的视频（按vod_time倒序）
            if ($rule === self::RULE_MERGE && empty($keepId)) {
                $keepId = $this->selectKeepId($ids, self::RULE_NEWEST);
            }

            if ($rule === self::RULE_MERGE && $keepId) {
                // 合并逻辑：将其他视频的播放源合并到保留的视频
                return $this->mergeVideos($keepId, array_values(array_diff($ids, [$keepId])));
            } else {
                // 删除逻辑：选择要保留的ID，删除其他
                $keepId = $this->selectKeepId($ids, $rule);
                $deleteIds = array_values(array_diff($ids, [$keepId]));
                
                if (empty($deleteIds)) {
                    return ['success' => false, 'message' => '没有需要删除的视频'];
                }
                
                $this->deleteVideos($deleteIds);
                return [
                    'success' => true,
                    'message' => '成功删除 ' . count($deleteIds) . ' 个重复视频',
                    'deleted_count' => count($deleteIds),
                    'keeped_id' => $keepId
                ];
            }
        } catch (Exception $e) {
            return ['success' => false, 'message' => '操作失败: ' . $e->getMessage()];
        }
    }

    /**
     * 合并视频 - 将删除视频的播放源合并到保留视频
     */
    private function mergeVideos($keepId, $deleteIds)
    {
        try {
            // 获取保留视频的信息
            $sql = "SELECT `vod_id`, `vod_play_from`, `vod_play_url`, `vod_play_server`, `vod_play_note` FROM `{$this->prefix}vod` WHERE `vod_id` = ?";
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute([$keepId]);
            $keepVideo = $stmt->fetch(PDO::FETCH_ASSOC);

            // 获取要删除视频的播放源
            $placeholders = implode(',', array_fill(0, count($deleteIds), '?'));
            $sql = "SELECT `vod_id`, `vod_play_from`, `vod_play_url`, `vod_play_server`, `vod_play_note` FROM `{$this->prefix}vod` WHERE `vod_id` IN ($placeholders)";
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute($deleteIds);
            $deleteVideos = $stmt->fetchAll(PDO::FETCH_ASSOC);

            // 苹果CMS默认分隔符
            $separator = '$$$';

            // 解析保留视频的播放数据
            $keepFroms = $this->explodeData($keepVideo['vod_play_from'], $separator);
            $keepUrls = $this->explodeData($keepVideo['vod_play_url'], $separator);
            $keepServers = $this->explodeData($keepVideo['vod_play_server'] ?? '', $separator);
            $keepNotes = $this->explodeData($keepVideo['vod_play_note'] ?? '', $separator);

            // 确保数组长度一致，避免后续索引越界
            $maxCount = count($keepFroms);
            $keepUrls = array_pad($keepUrls, $maxCount, '');
            $keepServers = array_pad($keepServers, $maxCount, '');
            $keepNotes = array_pad($keepNotes, $maxCount, '');

            foreach ($deleteVideos as $video) {
                $delFroms = $this->explodeData($video['vod_play_from'], $separator);
                $delUrls = $this->explodeData($video['vod_play_url'], $separator);
                $delServers = $this->explodeData($video['vod_play_server'] ?? '', $separator);
                $delNotes = $this->explodeData($video['vod_play_note'] ?? '', $separator);

                foreach ($delFroms as $index => $from) {
                    if (empty($from)) continue;

                    // 检查该播放器是否已存在于保留视频中
                    if (!in_array($from, $keepFroms)) {
                        $keepFroms[] = $from;
                        $keepUrls[] = $delUrls[$index] ?? '';
                        $keepServers[] = $delServers[$index] ?? '';
                        $keepNotes[] = $delNotes[$index] ?? '';
                    }
                }
            }

            // 合并回字符串
            $newPlayFrom = implode($separator, $keepFroms);
            $newPlayUrl = implode($separator, $keepUrls);
            $newPlayServer = implode($separator, $keepServers);
            $newPlayNote = implode($separator, $keepNotes);

            // 更新保留视频
            // 注意：这里尝试同时更新 server 和 note，如果表字段不存在可能会报错，但标准版都有
            // 为了安全起见，先检查字段是否存在太麻烦，直接尝试更新。如果之前查询成功，更新应该也没问题。
            $sql = "UPDATE `{$this->prefix}vod` SET 
                    `vod_play_from` = ?, 
                    `vod_play_url` = ?, 
                    `vod_play_server` = ?, 
                    `vod_play_note` = ? 
                    WHERE `vod_id` = ?";
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute([$newPlayFrom, $newPlayUrl, $newPlayServer, $newPlayNote, $keepId]);

            // 删除其他视频
            $this->deleteVideos($deleteIds);

            return [
                'success' => true,
                'message' => '成功合并 ' . count($deleteIds) . ' 个视频',
                'merged_count' => count($deleteIds),
                'keeped_id' => $keepId
            ];
        } catch (Exception $e) {
            return ['success' => false, 'message' => '合并失败: ' . $e->getMessage()];
        }
    }

    /**
     * 辅助方法：分割数据
     */
    private function explodeData($str, $separator)
    {
        if (empty($str)) return [];
        return explode($separator, $str);
    }

    /**
     * 根据规则选择要保留的视频ID
     */
    private function selectKeepId($ids, $rule)
    {
        $placeholders = implode(',', array_fill(0, count($ids), '?'));
        
        switch ($rule) {
            case self::RULE_NEWEST:
                $sql = "SELECT `vod_id` FROM `{$this->prefix}vod` WHERE `vod_id` IN ($placeholders) ORDER BY `vod_time` DESC LIMIT 1";
                break;
            case self::RULE_OLDEST:
                $sql = "SELECT `vod_id` FROM `{$this->prefix}vod` WHERE `vod_id` IN ($placeholders) ORDER BY `vod_time` ASC LIMIT 1";
                break;
            case self::RULE_MOST_PLAYED:
                $sql = "SELECT `vod_id` FROM `{$this->prefix}vod` WHERE `vod_id` IN ($placeholders) ORDER BY `vod_hits` DESC LIMIT 1";
                break;
            case self::RULE_HIGHEST_SCORE:
                $sql = "SELECT `vod_id` FROM `{$this->prefix}vod` WHERE `vod_id` IN ($placeholders) ORDER BY `vod_score` DESC LIMIT 1";
                break;
            default:
                return $ids[0];
        }
        
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($ids);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        return $result ? $result['vod_id'] : $ids[0];
    }

    /**
     * 删除视频
     */
    private function deleteVideos($ids)
    {
        $placeholders = implode(',', array_fill(0, count($ids), '?'));
        $sql = "DELETE FROM `{$this->prefix}vod` WHERE `vod_id` IN ($placeholders)";
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($ids);
    }

    /**
     * 备份视频数据
     */
    private function backupVideos($ids)
    {
        $placeholders = implode(',', array_fill(0, count($ids), '?'));
        $sql = "SELECT * FROM `{$this->prefix}vod` WHERE `vod_id` IN ($placeholders)";
        
        try {
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute($ids);
            $videos = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            $backupFile = BACKUP_PATH . date('Y-m-d_H-i-s') . '_backup.json';
            file_put_contents($backupFile, json_encode($videos, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT));
            
            return true;
        } catch (Exception $e) {
            // 备份失败不影响主操作
            return false;
        }
    }

    /**
     * 获取统计信息
     */
    public function getStats()
    {
        try {
            // 先检查表是否存在
            $checkSql = "SHOW TABLES LIKE '{$this->prefix}vod'";
            $checkStmt = $this->pdo->query($checkSql);
            
            if ($checkStmt->rowCount() == 0) {
                return ['error' => "表 {$this->prefix}vod 不存在，请确认数据库前缀是否正确"];
            }

            // 统计总数
            $totalSql = "SELECT COUNT(*) as `count` FROM `{$this->prefix}vod`";
            $totalStmt = $this->pdo->query($totalSql);
            $totalResult = $totalStmt->fetch(PDO::FETCH_ASSOC);
            $total = intval($totalResult['count'] ?? 0);

            // 统计重复数
            $duplicateSql = "SELECT COUNT(DISTINCT `vod_name`) as `count` FROM (
                            SELECT `vod_name`, COUNT(*) as `cnt` 
                            FROM `{$this->prefix}vod` 
                            WHERE `vod_name` != '' 
                            GROUP BY `vod_name` 
                            HAVING `cnt` > 1) as `t`";
            $dupStmt = $this->pdo->query($duplicateSql);
            $dupResult = $dupStmt->fetch(PDO::FETCH_ASSOC);
            $duplicateCount = intval($dupResult['count'] ?? 0);

            return [
                'total' => $total,
                'duplicate_count' => $duplicateCount,
                'unique_count' => $total - $duplicateCount
            ];
        } catch (Exception $e) {
            return ['error' => '统计失败: ' . $e->getMessage()];
        }
    }

    /**
     * 记录日志
     */
    public static function log($message)
    {
        $logFile = LOG_PATH . date('Y-m-d') . '.log';
        $logMessage = '[' . date('H:i:s') . '] ' . $message . PHP_EOL;
        file_put_contents($logFile, $logMessage, FILE_APPEND);
    }
}
?>
