<?php
namespace addons\xunsearch\controller;

use think\Controller;
use think\Db;
use addons\xunsearch\library\XsService;

class Admin extends Controller
{
public function index()
    {
        $xs = new XsService();
        $config = get_addon_config('xunsearch');
        
        // === 新增：防 PHP 8 模板 date() 类型错误补丁 ===
        if (is_array($config)) {
            foreach ($config as $k => $v) {
                // 如果 config 里有空字符串，强制转为 0，防止前端模板里的 date 函数接到字符串报错
                if ($v === '') {
                    $config[$k] = 0;
                }
            }
        }
        
        $stats = $this->getIndexStats();
        $this->assign('stats', $stats);
        $this->assign('config', $config);
        
        // 设置模板路径为 view 目录
        $this->view->config(['view_path' => __DIR__ . '/../view/']);
        return $this->fetch('admin/index');
    }
    
private function getIndexStats()
    {
        $stats = [
            'total_vod' => 0,
            'total_index' => 0,
            'last_update' => 0,
            'server_status' => 'unknown'
        ];
        
        try {
            $stats['total_vod'] = Db::name('vod')->count();
            $xs = new XsService();
            $search = $xs->getXs()->getSearch();
            $stats['total_index'] = $search->getDbTotal();
            $stats['server_status'] = 'connected';
            
            // 获取最后索引时间（从进度文件）
            $progress_file = __DIR__ . '/../rebuild_progress.txt';
            if (file_exists($progress_file)) {
                // === 新增：强制转换为 int 类型 ===
                $stats['last_update'] = (int)filemtime($progress_file);
            }
        } catch (\Exception $e) {
            $stats['server_status'] = 'error: ' . $e->getMessage();
        }
        
        return $stats;
    }
    
    /**
     * 强行解除讯搜的死锁 (Ajax接口版)
     */
    public function unlock()
    {
        try {
            $xs = new XsService();
            $index = $xs->getXs()->getIndex();
            // 发送强制停止重建的指令
            $index->stopRebuild();
            
            // 返回 JSON 格式给前端的 Layui 弹窗
            return json([
                'code' => 1,
                'msg' => '解锁成功'
            ]);
        } catch (\Exception $e) {
            return json([
                'code' => 0,
                'msg' => '解锁异常: ' . $e->getMessage()
            ]);
        }
    }
    
/**
     * 分批执行的重建索引方法
     */
    public function rebuild()
    {
        // 接收前端传来的页码，默认为第1页
        $page = input('page/d', 1);
        $limit = 500; // 每次处理 500 条
        
        try {
            $xs = new \addons\xunsearch\library\XsService();
            $index = $xs->getXs()->getIndex();
            $total = Db::name('vod')->count();
            
            $status_file = __DIR__ . '/../rebuild_status.json';
            
            if ($page == 1) {
                // 如果是第 1 页，发送开始重建指令（会清空临时库）
                $index->beginRebuild();
                file_put_contents($status_file, json_encode(['total' => $total, 'processed' => 0, 'percent' => 0]));
            }
            
            // 只查询当前这 500 条数据
            $list = Db::name('vod')
                ->field('vod_id,vod_name,type_id,vod_en,vod_sub,vod_actor,vod_director,vod_content,vod_time,vod_pic')
                ->page($page, $limit)
                ->select();
                
            if (!empty($list)) {
                foreach($list as $v) {
                    if(empty($v['vod_id']) || empty($v['vod_name'])) continue;
                    try {
                        $doc = new \XSDocument();
                        $doc->setFields([
                            'id' => $v['vod_id'],
                            'vod_name' => $v['vod_name'],
                            'type_id' => $v['type_id'] ?? 0,
                            'vod_en' => $v['vod_en'] ?? '',
                            'vod_sub' => $v['vod_sub'] ?? '',
                            'vod_actor' => $v['vod_actor'] ?? '',
                            'vod_director' => $v['vod_director'] ?? '',
                            'vod_content' => strip_tags($v['vod_content'] ?? ''),
                            'vod_time' => $v['vod_time'] ?? time(),
                            'vod_pic' => $v['vod_pic'] ?? ''
                        ]);
                        $index->add($doc);
                    } catch (\Exception $e) {
                        continue;
                    }
                }
            }
            
            // 计算最新进度
            $processed_so_far = (($page - 1) * $limit) + count($list);
            if ($processed_so_far > $total) $processed_so_far = $total;
            
            $percent = $total > 0 ? round(($processed_so_far / $total) * 100, 2) : 100;
            
            // 更新进度文件
            file_put_contents($status_file, json_encode([
                'total' => $total,
                'processed' => $processed_so_far,
                'percent' => $percent
            ]));
            
            // 判断是否全部处理完毕
            if (empty($list) || count($list) < $limit || $processed_so_far >= $total) {
                // 最后一页，发送结束重建指令（覆盖旧库，生效新数据）
                $index->endRebuild();
                return json(['code' => 1, 'msg' => '完成', 'data' => ['status' => 'done', 'total' => $processed_so_far]]);
            }
            
            // 数据还没跑完，告诉前端继续请求下一页
            return json([
                'code' => 1, 
                'msg' => '进行中', 
                'data' => [
                    'status' => 'continue', 
                    'next_page' => $page + 1, 
                    'percent' => $percent, 
                    'processed' => $processed_so_far, 
                    'total' => $total
                ]
            ]);
            
        } catch (\Exception $e) {
            return json(['code' => 0, 'msg' => '重建失败: ' . $e->getMessage()]);
        }
    }
    
    /**
     * 前端获取重建进度的专用接口
     */

    
/**
     * 分批执行的增量更新方法（带进度条支持）
     */
    public function update()
    {
        // 接收前端传来的页码，默认为第1页
        $page = input('page/d', 1);
        $limit = 500; // 每次处理 500 条
        
        // 为了防止分页过程中时间流逝导致数据错位，我们由第一页生成一个固定的时间戳，并一直往后传
        $time_limit = input('time_limit/d', 0);
        if ($time_limit == 0) {
            $time_limit = time() - 7200; // 默认获取过去 2 小时内更新的视频
        }
        
        try {
            $xs = new \addons\xunsearch\library\XsService();
            
            // 检查数据库中是否有 vod_time 字段
            $fields = Db::query("SHOW COLUMNS FROM " . config('database.prefix') . "vod LIKE 'vod_time'");
            
            if (empty($fields)) {
                // 如果没有时间字段，作为降级方案，只查最新的 100 条
                if ($page > 1) {
                    return json(['code' => 1, 'msg' => '完成', 'data' => ['status' => 'done', 'total' => 100]]);
                }
                $list = Db::name('vod')
                    ->order('vod_id', 'desc')
                    ->limit(100)
                    ->field('vod_id,vod_name,type_id,vod_en,vod_sub,vod_actor,vod_director,vod_content,vod_time,vod_pic')
                    ->select();
                $total = count($list);
            } else {
                // 有时间字段，标准增量更新流程
                $where = [];
                $where['vod_time'] = ['>', $time_limit];
                $total = Db::name('vod')->where($where)->count();
                
                $list = Db::name('vod')
                    ->where($where)
                    ->field('vod_id,vod_name,type_id,vod_en,vod_sub,vod_actor,vod_director,vod_content,vod_time,vod_pic')
                    ->page($page, $limit)
                    ->select();
            }
            
            if (!empty($list)) {
                foreach($list as $v) {
                    // 增量更新使用 updateIndex（内部逻辑：如果存在则修改，不存在则新增）
                    $xs->updateIndex($v);
                }
            }
            
            // 计算最新进度
            $processed_so_far = (($page - 1) * $limit) + count($list);
            if ($processed_so_far > $total) $processed_so_far = $total;
            
            $percent = $total > 0 ? round(($processed_so_far / $total) * 100, 2) : 100;
            
            // 判断是否全部处理完毕
            if (empty($list) || count($list) < $limit || $processed_so_far >= $total) {
                // 更新一下最后索引时间的文本记录
                $progress_file = __DIR__ . '/../rebuild_progress.txt';
                if (file_exists($progress_file)) {
                    $progress = date('Y-m-d H:i:s') . ' - 增量更新: 更新了 ' . $processed_so_far . ' 条数据' . PHP_EOL;
                    file_put_contents($progress_file, $progress, FILE_APPEND);
                }
                
                return json(['code' => 1, 'msg' => '完成', 'data' => ['status' => 'done', 'total' => $processed_so_far]]);
            }
            
            // 数据还没跑完，告诉前端继续请求下一页，并把固定好的时间戳传下去
            return json([
                'code' => 1, 
                'msg' => '进行中', 
                'data' => [
                    'status' => 'continue', 
                    'next_page' => $page + 1, 
                    'time_limit' => $time_limit, // 核心：保持时间窗口一致
                    'percent' => $percent, 
                    'processed' => $processed_so_far, 
                    'total' => $total
                ]
            ]);
            
        } catch (\Exception $e) {
            return json(['code' => 0, 'msg' => '更新失败: ' . $e->getMessage()]);
        }
    }
    
    public function test()
    {
        $keyword = input('keyword', '');
        
        if(empty($keyword)) {
            return json(['code' => 0, 'msg' => '请输入搜索关键词']);
        }
        
        try {
            $xs = new XsService();
            $res = $xs->search($keyword, 1, 10);
            $result = [
                'code' => 1,
                'msg' => '搜索成功',
                'data' => $res
            ];
        } catch (\Exception $e) {
            $result = [
                'code' => 0,
                'msg' => '搜索失败: ' . $e->getMessage()
            ];
        }
        
        return json($result);
    }
    
    public function status()
    {
        try {
            $stats = $this->getIndexStats();
            $result = [
                'code' => 1,
                'msg' => '获取成功',
                'data' => $stats
            ];
        } catch (\Exception $e) {
            $result = [
                'code' => 0,
                'msg' => '获取失败: ' . $e->getMessage()
            ];
        }
        
        return json($result);
    }
    
    public function tvbox()
    {
        $wd = input('wd', '');
        $page = input('page', 1);
        
        if(empty($wd)) {
            return $this->returnTVBoxError('请输入搜索关键词');
        }
        
        try {
            $xs = new XsService();
            $res = $xs->search($wd, $page, 20);
            
            $videos = [];
            foreach($res['list'] as $item) {
                $videos[] = [
                    'vod_id' => $item['vod_id'],
                    'vod_name' => $item['vod_name'],
                    'type_id' => $item['type_id'],
                    'vod_pic' => $item['vod_pic'] ?? '',
                    'vod_time' => $item['vod_time']
                ];
            }
            
            $result = [
                'code' => 1,
                'msg' => '搜索成功',
                'data' => [
                    'list' => $videos,
                    'total' => $res['total'],
                    'page' => $page,
                    'pagecount' => ceil($res['total'] / 20)
                ]
            ];
            
        } catch (\Exception $e) {
            $result = $this->returnTVBoxError('搜索失败: ' . $e->getMessage());
        }
        
        header('Content-Type: application/json; charset=utf-8');
        echo json_encode($result, JSON_UNESCAPED_UNICODE);
        exit;
    }
    
    public function api()
    {
        $wd = input('wd', '');
        $page = input('page', 1);
        $limit = input('limit', 20);
        $callback = input('callback', '');
        
        if(empty($wd)) {
            $result = [
                'code' => 0,
                'msg' => '请输入搜索关键词'
            ];
        } else {
            try {
                $xs = new XsService();
                $res = $xs->search($wd, $page, $limit);
                
                $result = [
                    'code' => 1,
                    'msg' => 'success',
                    'data' => [
                        'list' => $res['list'],
                        'total' => $res['total'],
                        'page' => $page,
                        'pagecount' => ceil($res['total'] / $limit)
                    ]
                ];
            } catch (\Exception $e) {
                $result = [
                    'code' => 0,
                    'msg' => $e->getMessage()
                ];
            }
        }
        
        if(!empty($callback)) {
            header('Content-Type: application/javascript; charset=utf-8');
            echo $callback . '(' . json_encode($result, JSON_UNESCAPED_UNICODE) . ')';
        } else {
            header('Content-Type: application/json; charset=utf-8');
            echo json_encode($result, JSON_UNESCAPED_UNICODE);
        }
        exit;
    }
    
    private function returnTVBoxError($msg)
    {
        return [
            'code' => 0,
            'msg' => $msg,
            'data' => ['list' => [], 'total' => 0]
        ];
    }
}