<?php

namespace app\api\controller;

use think\Cache;
use think\Db;
use think\Request;

/**
 * 搜索
 */
class Appfox extends Base
{
    var $_param;

    public function __construct()
    {
        parent::__construct();
        $this->_param = input('', '', 'trim,urldecode');
    }

    /**
     * 极速索引直查 + 讯搜混合引擎 (抗CC死锁版)
     */
    private function searchIdsWithXunsearch($keyword)
    {
        // 🛡️ 1. 物理级防CC：超长乱码直接丢弃，不给数据库任何压力
        if (mb_strlen($keyword, 'utf-8') > 20) {
            return [];
        }
        
        $exact_ids = [];
        // 🚀 2. O(1) 极速索引直查：专门解决讯搜分词切坏日韩剧的问题！
        // 用 "=" 和 "like 'xxx%'" 完美触发 MySQL 的 vod_name 索引，0.001秒内返回，绝不死锁！
        try {
            // 完全相等，强行置顶
            $exact_ids = Db::name('vod')->where('vod_name', '=', $keyword)->column('vod_id') ?: [];
            
            // 前缀相等（解决用户少打字的情况，且不引发全表扫描）
            $prefix_ids = Db::name('vod')->where('vod_name', 'like', $keyword . '%')->limit(30)->column('vod_id') ?: [];
            
            $exact_ids = array_unique(array_merge($exact_ids, $prefix_ids));
        } catch (\Exception $e) {}

        // 🔍 3. 讯搜模糊兜底
        $fuzzy_ids = [];
        try {
            $xs_path = dirname(dirname(dirname(dirname(__DIR__)))) . '/addons/xunsearch/library/XsService.php';
            if (file_exists($xs_path)) {
                require_once $xs_path;
                if (class_exists('addons\xunsearch\library\XsService')) {
                    $xs = new \addons\xunsearch\library\XsService();
                    $result = $xs->search($keyword, 1, 100);

                    if ($result && $result['total'] > 0) {
                        foreach ($result['list'] as $v) {
                            if (!empty($v['vod_id'])) {
                                $fuzzy_ids[] = $v['vod_id'];
                            }
                        }
                    }
                }
            }
        } catch (\Exception $e) {}

        // 🏆 4. 合并并输出：极速直查出的《与恶魔有约》死死钉在第 1 名！
        $final_ids = array_unique(array_merge($exact_ids, $fuzzy_ids));
        return array_slice($final_ids, 0, 100);
    }

    protected $simple_vod_fields = "vod_id, vod_name, vod_pic, vod_remarks, vod_actor, vod_blurb, vod_pic_slide";

    protected function replaceVodPic($vod_list)
    {
        foreach ($vod_list as &$vod) {
            $vod['vod_pic'] = $this->getImgUrl($vod['vod_pic'] ?? '');
        }
        return $vod_list;
    }


    private function getInitVodWeekList()
    {
        $week_list = [];
        $weekdays = ['一', '二', '三', '四', '五', '六', '日'];
        for ($week = 1; $week <= 7; $week++) {
            $week_list[] = [
                'week_name' => $weekdays[$week - 1],
                'week_num' => $week,
                'week_list' => []
            ];
        }
        return $week_list;
    }

    private function getVodWeekList($week = 1, $page = 1)
    {
        $weekdays = ['一', '二', '三', '四', '五', '六', '日'];
        $where = ['vod_status' => 1];

        $cn_week = $weekdays[$week - 1];
        $order = 'vod_time desc';
        $week_list = Db::name('vod')
            ->field($this->simple_vod_fields)
            ->where($where)
            ->where(function ($query) use ($cn_week, $week) {
                $query->where('vod_weekday', 'like', "%{$week}%")->whereor('vod_weekday', 'like', "%{$cn_week}%");
            })
            ->order($order)
            ->page($page, 30)
            ->select();
        return $this->replaceVodPic($week_list);
    }


    public function vod_json($res)
    {
        $type_list = model('Type')->getCache('type_list');
        foreach ($res['list'] as $k => &$v) {
            $type_info = $type_list[$v['type_id']] ?? ['type_name'=>'未知'];
            $v['type_name'] = $type_info['type_name'] ?? '未知';
            
            $v['vod_time'] = date('Y-m-d H:i:s', (int)$v['vod_time']);

            if (substr($v["vod_pic"], 0, 4) == "mac:") {
                $v["vod_pic"] = str_replace('mac:', $this->getImgUrlProtocol('vod'), $v["vod_pic"]);
            } elseif (!empty($v["vod_pic"]) && substr($v["vod_pic"], 0, 4) != "http" && substr($v["vod_pic"], 0, 2) != "//") {
                $v["vod_pic"] = ($GLOBALS['config']['api']['vod']['imgurl'] ?? '') . $v["vod_pic"];
            }

            if (($this->_param['ac'] ?? '') == 'videolist' || ($this->_param['ac'] ?? '') == 'detail') {
                if (!empty($GLOBALS['config']['api']['vod']['from'])) {
                    $arr_from = explode('$$$', $v['vod_play_from'] ?? '');
                    $arr_url = explode('$$$', $v['vod_play_url'] ?? '');
                    $arr_server = explode('$$$', $v['vod_play_server'] ?? '');
                    $arr_note = explode('$$$', $v['vod_play_note'] ?? '');
                    $vod_play_from_list = explode(',', trim($GLOBALS['config']['api']['vod']['from']));
                    $vod_play_from_list = array_unique($vod_play_from_list);
                    $vod_play_from_list = array_filter($vod_play_from_list);
                    $vod_play_url_list = [];
                    $vod_play_server_list = [];
                    $vod_play_note_list = [];
                    foreach ($vod_play_from_list as $vod_play_from_index => $vod_play_from) {
                        $key = array_search($vod_play_from, $arr_from);
                        if ($key === false) {
                            unset($vod_play_from_list[$vod_play_from_index]);
                            continue;
                        }
                        $vod_play_url_list[] = $arr_url[$key] ?? '';
                        $vod_play_server_list[] = $arr_server[$key] ?? '';
                        $vod_play_note_list[] = $arr_note[$key] ?? '';
                    }
                    $res['list'][$k]['vod_play_from'] = join(',', $vod_play_from_list);
                    $res['list'][$k]['vod_play_url'] = join('$$$', $vod_play_url_list);
                    $res['list'][$k]['vod_play_server'] = join('$$$', $vod_play_server_list);
                    $res['list'][$k]['vod_play_note'] = join('$$$', $vod_play_note_list);
                }
            } else {
                if (!empty($GLOBALS['config']['api']['vod']['from'])) {
                    $arr_from = explode('$$$', $v['vod_play_from'] ?? '');
                    $vod_play_from_list = explode(',', trim($GLOBALS['config']['api']['vod']['from']));
                    $vod_play_from_list = array_unique($vod_play_from_list);
                    $vod_play_from_list = array_filter($vod_play_from_list);
                    foreach ($vod_play_from_list as $vod_play_from_index => $vod_play_from) {
                        $key = array_search($vod_play_from, $arr_from);
                        if ($key === false) {
                            unset($vod_play_from_list[$vod_play_from_index]);
                            continue;
                        }
                    }
                    $res['list'][$k]['vod_play_from'] = join(',', $vod_play_from_list);
                } else {
                    $res['list'][$k]['vod_play_from'] = str_replace('$$$', ',', $v['vod_play_from'] ?? '');
                }
            }
        }


        if (($this->_param['ac'] ?? '') != 'videolist' && ($this->_param['ac'] ?? '') != 'detail') {
            $class = [];
            $typefilter = explode(',', $GLOBALS['config']['api']['vod']['typefilter'] ?? '');

            foreach ($type_list as $k => &$v) {
                if (!empty($GLOBALS['config']['api']['vod']['typefilter'])) {
                    if (in_array($v['type_id'], $typefilter)) {
                        $class[] = ['type_id' => $v['type_id'], 'type_pid' => $v['type_pid'], 'type_name' => $v['type_name']];
                    }
                } else {
                    $class[] = ['type_id' => $v['type_id'], 'type_pid' => $v['type_pid'], 'type_name' => $v['type_name']];
                }
            }
            $res['class'] = $class;
        }
        return $res;
    }

    public function vod_xml($res)
    {
        $xml = '<?xml version="1.0" encoding="utf-8"?>';
        $xml .= '<rss version="5.1">';
        $type_list = model('Type')->getCache('type_list');

        $xml .= '<list page="' . ($res['page']??1) . '" pagecount="' . ($res['pagecount']??1) . '" pagesize="' . ($res['limit']??20) . '" recordcount="' . ($res['total']??0) . '">';
        foreach ($res['list'] as $k => &$v) {
            $type_info = $type_list[$v['type_id']] ?? ['type_name'=>'未知'];
            $xml .= '<video>';
            $xml .= '<last>' . date('Y-m-d H:i:s', (int)$v['vod_time']) . '</last>';
            $xml .= '<id>' . $v['vod_id'] . '</id>';
            $xml .= '<tid>' . $v['type_id'] . '</tid>';
            $xml .= '<name><![CDATA[' . $v['vod_name'] . ']]></name>';
            $xml .= '<type>' . $type_info['type_name'] . '</type>';
            if (substr($v["vod_pic"]??'', 0, 4) == "mac:") {
                $v["vod_pic"] = str_replace('mac:', $this->getImgUrlProtocol('vod'), $v["vod_pic"]);
            } elseif (!empty($v["vod_pic"]) && substr($v["vod_pic"], 0, 4) != "http" && substr($v["vod_pic"], 0, 2) != "//") {
                $v["vod_pic"] = ($GLOBALS['config']['api']['vod']['imgurl']??'') . $v["vod_pic"];
            }

            if (($this->_param['ac']??'') == 'videolist' || ($this->_param['ac']??'') == 'detail') {
                $tempurl = $this->vod_url_deal($v["vod_play_url"]??'', $v["vod_play_from"]??'', $GLOBALS['config']['api']['vod']['from']??'', 'xml');

                $xml .= '<pic>' . ($v["vod_pic"]??'') . '</pic>';
                $xml .= '<lang>' . ($v['vod_lang']??'') . '</lang>';
                $xml .= '<area>' . ($v['vod_area']??'') . '</area>';
                $xml .= '<year>' . ($v['vod_year']??'') . '</year>';
                $xml .= '<state>' . ($v['vod_serial']??'') . '</state>';
                $xml .= '<note><![CDATA[' . ($v['vod_remarks']??'') . ']]></note>';
                $xml .= '<actor><![CDATA[' . ($v['vod_actor']??'') . ']]></actor>';
                $xml .= '<director><![CDATA[' . ($v['vod_director']??'') . ']]></director>';
                $xml .= '<dl>' . $tempurl . '</dl>';
                $xml .= '<des><![CDATA[' . ($v['vod_content']??'') . ']]></des>';
            } else {
                if (($GLOBALS['config']['api']['vod']['from']??'') != '') {
                    $xml .= '<dt>' . $GLOBALS['config']['api']['vod']['from'] . '</dt>';
                } else {
                    $xml .= '<dt>' . str_replace('$$$', ',', $v['vod_play_from']??'') . '</dt>';
                }
                $xml .= '<note><![CDATA[' . ($v['vod_remarks']??'') . ']]></note>';
            }
            $xml .= '</video>';
        }
        $xml .= '</list>';

        if (($this->_param['ac']??'') != 'videolist' && ($this->_param['ac']??'') != 'detail') {
            $xml .= "<class>";
            $typefilter = explode(',', $GLOBALS['config']['api']['vod']['typefilter']??'');
            foreach ($type_list as $k => &$v) {
                if (($v['type_mid']??0) == 1) {
                    if (!empty($GLOBALS['config']['api']['vod']['typefilter'])) {
                        if (in_array($v['type_id'], $typefilter)) {
                            $xml .= "<ty id=\"" . $v["type_id"] . "\">" . $v["type_name"] . "</ty>";
                        }
                    } else {
                        $xml .= "<ty id=\"" . $v["type_id"] . "\">" . $v["type_name"] . "</ty>";
                    }
                }
            }
            $xml .= "</class>";
        }
        $xml .= "</rss>";
        return $xml;
    }


    protected function getImgUrl($img_path)
    {
        $config = config('maccms');
        $upload_config = $config['upload'] ?? [];
        $protocol = $upload_config['protocol'] ?? 'http';
        $img_path = str_replace("mac://", "{$protocol}://", $img_path);
        if (!stristr($img_path, "http")) {
            if (stristr($img_path, "//")) {
                $img_path = "http:" . $img_path;
            } else {
                $img_path = \request()->domain() . "/" . ltrim($img_path, '/');
            }
        }
        return $img_path;
    }

    protected function setData($data = [], $msg = '', $code = 200)
    {
        return json(['data' => $data, 'msg' => $msg, 'code' => $code]);
    }

    public function vod()
    {
        if (($GLOBALS['config']['api']['vod']['charge'] ?? 0) == 1) {
            $h = $_SERVER['REMOTE_ADDR'] ?? '';
            if (!$h) {
                echo lang('api/auth_err');
                exit;
            } else {
                $auth = $GLOBALS['config']['api']['vod']['auth'] ?? '';
                $this->checkDomainAuth($auth);
            }
        }

        $cache_time = intval($GLOBALS['config']['api']['vod']['cachetime'] ?? 0);
        $cach_name = ($GLOBALS['config']['app']['cache_flag'] ?? 'mc') . '_' . 'api_vod_' . md5(http_build_query($this->_param));
        $html = Cache::get($cach_name);
        
        if (empty($html) || $cache_time == 0) {
            $where = [];
            if (!empty($this->_param['ids'])) {
                $where['vod_id'] = ['in', $this->_param['ids']];
            }
            if (!empty($GLOBALS['config']['api']['vod']['typefilter'])) {
                $where['type_id'] = ['in', $GLOBALS['config']['api']['vod']['typefilter']];
            }

            if (!empty($this->_param['t'])) {
                if (empty($GLOBALS['config']['api']['vod']['typefilter']) || strpos($GLOBALS['config']['api']['vod']['typefilter'], $this->_param['t']) !== false) {
                    $where['type_id'] = $this->_param['t'];
                }
            }
            if (isset($this->_param['isend'])) {
                $where['vod_isend'] = $this->_param['isend'] == 1 ? 1 : 0;
            }
            if (!empty($this->_param['h'])) {
                $todaydate = date('Y-m-d', strtotime('+1 days'));
                $tommdate = date('Y-m-d H:i:s', strtotime('-' . $this->_param['h'] . ' hours'));
                $where['vod_time'] = [['gt', strtotime($tommdate)], ['lt', strtotime($todaydate)]];
            }
            
            $custom_order = null;
            
            if (!empty($this->_param['wd'])) {
                if (empty($this->_param['pg'])) $this->_param['pg'] = 1;
                $pagesize = $GLOBALS['config']['api']['vod']['pagesize'] ?? 20;
                if (!empty($this->_param['pagesize']) && $this->_param['pagesize'] > 0) {
                    $pagesize = min((int)$this->_param['pagesize'], 100);
                }
                
                $keyword = urldecode($this->_param['wd']);
                $keyword = strip_tags(trim($keyword));
                
                if (!empty($keyword)) {
                    // 🛡️ 强制屏蔽所有 LIKE 全表扫描查询，完美保护 CPU！
                    $xs_ids = $this->searchIdsWithXunsearch($keyword);
                    
                    if (!empty($xs_ids)) {
                        $where['vod_id'] = ['in', $xs_ids];
                        $custom_order = Db::raw("field(vod_id," . implode(',', $xs_ids) . ")");
                    } else {
                        // 没查到直接强制短路，绝不抛给数据库全表扫描
                        $where['vod_id'] = -1; 
                    }
                }
            }
            
            if (!empty($this->_param['year'])) {
                $param_year = trim($this->_param['year']);
                if (strlen($param_year) == 4) {
                    $year = intval($param_year);
                } elseif (strlen($param_year) == 9) {
                    $start = (int)substr($param_year, 0, 4);
                    $end = (int)substr($param_year, 5, 4);
                    if ($start > $end) {
                        $tmp_num = $end; $end = $start; $start = $tmp_num;
                    }
                    $tmp_arr = [];
                    $start = max($start, 1900);
                    $end = min($end, date('Y') + 3);
                    for ($i = $start; $i <= $end; $i++) $tmp_arr[] = $i;
                    $year = join(',', $tmp_arr);
                }
                $where['vod_year'] = ['in', explode(',', $year ?? '')];
            }
            
            if (empty($GLOBALS['config']['api']['vod']['from']) && !empty($this->_param['from']) && strlen($this->_param['from']) >= 2) {
                $GLOBALS['config']['api']['vod']['from'] = $this->_param['from'];
            }
            if (!empty($GLOBALS['config']['api']['vod']['from'])) {
                $vod_play_from_list = explode(',', trim($GLOBALS['config']['api']['vod']['from']));
                $vod_play_from_list = array_unique($vod_play_from_list);
                $vod_play_from_list = array_filter($vod_play_from_list);
                if (!empty($vod_play_from_list)) {
                    $where['vod_play_from'] = ['or'];
                    foreach ($vod_play_from_list as $vod_play_from) {
                        array_unshift($where['vod_play_from'], ['like', '%' . trim($vod_play_from) . '%']);
                    }
                }
            }
            if (!empty($GLOBALS['config']['api']['vod']['datafilter'])) {
                $where['_string'] .= ' ' . $GLOBALS['config']['api']['vod']['datafilter'];
            }
            
            $pg = empty($this->_param['pg']) ? 1 : max(1, (int)$this->_param['pg']);
            $pagesize = $GLOBALS['config']['api']['vod']['pagesize'] ?? 20;
            if (!empty($this->_param['pagesize']) && $this->_param['pagesize'] > 0) {
                $pagesize = min((int)$this->_param['pagesize'], 100);
            }

            $sort_direction = (!empty($this->_param['sort_direction']) && $this->_param['sort_direction'] == 'asc') ? 'asc' : 'desc';
            $order = $custom_order ? $custom_order : 'vod_time ' . $sort_direction;
            
            $field = 'vod_id,vod_name,type_id,"" as type_name,vod_en,vod_time,vod_remarks,vod_play_from,vod_pic,vod_score,vod_hits,vod_actor,vod_blurb';

            if (($this->_param['ac'] ?? '') == 'videolist' || ($this->_param['ac'] ?? '') == 'detail') {
                $field = '*';
            }
            
            $res = model('vod')->listData($where, $order, $pg, $pagesize, 0, $field, 0);

            // 🔴 预防 N+1 查询风暴，一次性聚合拉取评论数
            $comment_counts = [];
            if (!empty($res['list'])) {
                $vod_ids = array_column($res['list'], 'vod_id');
                if (!empty($vod_ids)) {
                    try {
                        $comment_counts = Db::name('comment')
                            ->where('comment_mid', 1)
                            ->where('comment_status', 1)
                            ->where('comment_rid', 'in', $vod_ids)
                            ->group('comment_rid')
                            ->column('count(comment_id)', 'comment_rid');
                    } catch (\Exception $e) {}
                }
            }

            $playerConfig = config('vodplayer');
            $enabledPlayers = [];

            if (is_array($playerConfig)) {
                foreach ($playerConfig as $code => $info) {
                    if (isset($info['status']) && $info['status'] == 1) {
                        $sort = isset($info['sort']) && is_numeric($info['sort']) ? (int)$info['sort'] : 0;
                        $enabledPlayers[$code] = [
                            'sort' => $sort,
                            'config' => $info
                        ];
                    }
                }
            }

            uasort($enabledPlayers, function ($a, $b) {
                if ($a['sort'] != $b['sort']) return $b['sort'] <=> $a['sort'];
                return 0;
            });

            $sortedPlayerCodes = array_keys($enabledPlayers);

            if (!empty($res['list']) && !empty($sortedPlayerCodes)) {
                foreach ($res['list'] as &$item) {
                    $playSources = [];

                    if (!empty($item['vod_play_from'])) {
                        $froms = explode('$$$', (string)$item['vod_play_from']);
                        $urls = !empty($item['vod_play_url']) ? explode('$$$', (string)$item['vod_play_url']) : [];

                        for ($i = 0; $i < count($froms); $i++) {
                            $code = $froms[$i];
                            $playSources[$code] = $urls[$i] ?? '';
                        }
                    }

                    $orderedFroms = [];
                    $orderedUrls = [];

                    foreach ($sortedPlayerCodes as $playerCode) {
                        if (isset($playSources[$playerCode])) {
                            $orderedFroms[] = $playerCode;
                            $orderedUrls[] = $playSources[$playerCode];
                            unset($playSources[$playerCode]);
                        }
                    }

                    foreach ($playSources as $code => $url) {
                        $orderedFroms[] = $code;
                        $orderedUrls[] = $url;
                    }

                    $item['vod_play_from'] = implode('$$$', $orderedFroms);
                    $item['vod_play_url'] = implode('$$$', $orderedUrls);
                    
                    // 0数据库压力，极速读取聚合好的评论数
                    $item['comment_count'] = isset($comment_counts[$item['vod_id']]) ? (int)$comment_counts[$item['vod_id']] : 0;
                }
                unset($item);
            }

            if (($this->_param['at'] ?? '') == 'xml') {
                $html = $this->vod_xml($res);
            } else {
                $html = json_encode($this->vod_json($res), JSON_UNESCAPED_UNICODE);
            }
            if ($cache_time > 0) {
                Cache::set($cach_name, $html, $cache_time);
            }
        }
        
        echo $html;
        exit;
    }


    public function index()
    {
        try {
            $mainCategories = Db::name('Type')
                ->where('type_mid', 1)
                ->where('type_pid', 0)
                ->order('type_sort ASC')
                ->select() ?: [];
        } catch (\Exception $e) { $mainCategories = []; }

        try {
            $pageSettings = Db::table('app_page_setting')->select() ?: [];
        } catch (\Exception $e) { $pageSettings = []; }

        $result = [
            'code' => 200,
            'msg' => '成功',
            'data' => []
        ];

        foreach ($mainCategories as $category) {
            $typeEn = $category['type_en'];
            $categoryData = [
                'navigationId' => $category['type_id'],
                'navigationName' => $category['type_name'],
                'banner' => [],
                'categories' => []
            ];

            $categorySettings = [];
            foreach ($pageSettings as $setting) {
                if (($setting['type_en'] ?? '') == $typeEn) $categorySettings[] = $setting;
            }

            foreach ($categorySettings as $setting) {
                if (($setting['type_id'] ?? '') == 0 && !empty($setting['carousel_recommend'])) {
                    $recommendLevel = (int)str_replace('recommend', '', $setting['carousel_recommend']);
                    $bannerVideos = Db::name('Vod')
                        ->where('type_id|type_id_1', $category['type_id'])
                        ->where('vod_level', $recommendLevel)
                        ->select() ?: [];
                    foreach ($bannerVideos as $video) {
                        $categoryData['banner'][] = $this->formatVideoData($video);
                    }
                    break; 
                }
            }

            foreach ($categorySettings as $setting) {
                if (($setting['type_id'] ?? '') == 0 && !empty($setting['today_recommend'])) {
                    $recommendLevel = (int)str_replace('recommend', '', $setting['today_recommend']);
                    $todayVideos = Db::name('Vod')
                        ->where('type_id|type_id_1', $category['type_id'])
                        ->where('vod_level', $recommendLevel)
                        ->select() ?: [];
                    $formattedVideos = [];
                    foreach ($todayVideos as $video) {
                        $formattedVideos[] = $this->formatVideoData($video);
                    }
                    $categoryData['categories'][] = [
                        'id' => 0, 
                        'name' => '今日推荐',
                        'videos' => $formattedVideos
                    ];
                    break;
                }
            }

            foreach ($categorySettings as $setting) {
                if (($setting['type_id'] ?? '') == 1 && !empty($setting['custom_recommend'])) {
                    $recommendLevel = (int)str_replace('recommend', '', $setting['custom_recommend']);
                    $customVideos = Db::name('Vod')
                        ->where('type_id|type_id_1', $category['type_id'])
                        ->where('vod_level', $recommendLevel)
                        ->select() ?: [];
                    $formattedVideos = [];
                    foreach ($customVideos as $video) {
                        $formattedVideos[] = $this->formatVideoData($video);
                    }
                    $categoryData['categories'][] = [
                        'id' => $setting['id'] ?? 0,
                        'name' => $setting['block_name'] ?? '板块',
                        'videos' => $formattedVideos
                    ];
                }
            }

            $result['data'][] = $categoryData;
        }

        return json($result);
    }

    private function formatVideoData($video)
    {
        return [
            'vod_id' => $video['vod_id'] ?? 0,
            'vod_name' => $video['vod_name'] ?? '',
            'vod_pic' => $video['vod_pic'] ?? '',
            'vod_remarks' => $video['vod_remarks'] ?? '',
            'vod_actor' => $video['vod_actor'] ?? '',
            'vod_blurb' => $video['vod_blurb'] ?? '',
            'vod_pic_slide' => $video['vod_pic_slide'] ?? ''
        ];
    }


    public function Config()
    {
        try {
            $adConfig = Db::table('appfox_advert_config')->find() ?: [];
            $systemConfig = Db::table('appfox_system_config')->find() ?: [];
            $searchEnable = Db::table('appfox_config')->where('name', 'search_collect_enable')->value('value') ?: 0;
            $searchList = Db::table('appfox_search_collect')->where('status', 1)->order('id', 'desc')->select() ?: [];
            $parseList = Db::table('appfox_parse_config')->where('status', 1)->order('id', 'desc')->select() ?: [];
        } catch (\Exception $e) {
            $adConfig = []; $systemConfig = []; $searchEnable = 0; $searchList = []; $parseList = [];
        }

        $playerParseList = config("vodplayer") ?: [];
        $playerList = [];
        foreach ($playerParseList as $playerCode => $playerInfo) {
            if (($playerInfo['status'] ?? 0) == 1) { 
                $playerList[] = [
                    'playerCode' => $playerCode,
                    'playerName' => $playerInfo['show'] ?? ''
                ];
            }
        }

        $http_type = ((isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] == 'on') || (isset($_SERVER['HTTP_X_FORWARDED_PROTO']) && $_SERVER['HTTP_X_FORWARDED_PROTO'] == 'https')) ? 'https://' : 'http://';

        $result = [
            'code' => 200,
            'msg' => '成功',
            'data' => [
                'userId' => $adConfig['user_id'] ?? 0,
                'enableSplash' => (bool)($adConfig['ad_splash_status'] ?? false),
                'enableInterstitial' => (bool)($adConfig['ad_home_page_insert_status'] ?? false),
                'enableHomeBanner' => (bool)($adConfig['ad_home_banner_status'] ?? false),
                'enableRankFeed' => (bool)($adConfig['ad_rank_feed_status'] ?? false),
                'enableHistoryFeed' => (bool)($adConfig['ad_history_feed_status'] ?? false),
                'enableSearchBanner' => (bool)($adConfig['ad_search_page_banner_status'] ?? false),
                'enableSearchResultFeed' => (bool)($adConfig['ad_search_feed_status'] ?? false),
                'enableMyBanner' => (bool)($adConfig['ad_mine_page_banner_status'] ?? false),
                'enablePlayerFeed' => (bool)($adConfig['ad_play_feed_status'] ?? false),
                'enablePlayerRewarded' => (bool)($adConfig['enablePlayerRewarded'] ?? false),
                'detailstatus' => (bool)($adConfig['detailstatus'] ?? false),
                'danmUrl' => $systemConfig['danmu_settings'] ?? '',
                'globalVideoDataUrl' => $http_type . ($_SERVER['HTTP_HOST'] ?? '') . "/api.php/provide/vod/",
                'shareText' => $systemConfig['share_text'] ?? '',
                'bottomNva' => $systemConfig['bottom_nav'] ?? '',
                'notice' => $systemConfig['public_settings'] ?? '',
                'shadingLayer' => $systemConfig['shading_layer'] ?? '',
                'app_sign' => $systemConfig['app_sign'] ?? '',
                'app_key' => $systemConfig['app_key'] ?? '',
                'openSearch' => (bool)$searchEnable,
                'playerList' => $playerList,
                'jiexiDataList' => array_map(function ($item) {
                    return [
                        'url' => $item['parse_url'] ?? '',
                        'name' => $item['parse_name'] ?? '',
                        'playerCode' => $item['player_code'] ?? '',
                        'directConcat' => (bool)($item['is_direct'] ?? false),
                        'playerType' => $item['player_type'] ?? 'IJK'
                    ];
                }, $parseList),
                'searchDataList' => array_map(function ($item) {
                    return [
                        'id' => $item['id'] ?? 0,
                        'name' => $item['search_name'] ?? '',
                        'url' => $item['search_url'] ?? '',
                        'sortOrder' => $item['sort_order'] ?? null
                    ];
                }, $searchList)
            ]
        ];

        return json($result);
    }


    public function gbook()
    {
        $data = input('post.');

        if (empty($data['gbook_content'])) {
            return json(['code' => 0, 'msg' => '留言内容不能为空']);
        }

        $data['gbook_name'] = '游客';
        $data['gbook_content'] = htmlspecialchars($data['gbook_content']);
        $data['gbook_time'] = time();
        $data['status'] = 1; 
        $data['ip'] = request()->ip();

        try {
            $result = Db::table('appfox_gbook')->insert($data);
            if ($result) return json(['code' => 1, 'msg' => '留言提交成功']);
        } catch (\Exception $e) {}
        
        return json(['code' => 0, 'msg' => '留言提交失败']);
    }


    public function update()
    {
        try {
            $currentVersion = input('currentVersionCode/d', 0);
            $userId         = input('userId/d', 0);

            $latestVersion = Db::table('appfox_version')
                ->order('version', 'desc')
                ->find();

            if (!$latestVersion) {
                return json(['code' => 404, 'msg'  => '没有找到版本信息', 'data' => null], 200, [], JSON_UNESCAPED_SLASHES);
            }

            $forceUpdateRequired = (
                $currentVersion < (int)($latestVersion['version'] ?? 0) &&
                (int)($latestVersion['is_force'] ?? 0) === 1
            );

            $releaseTime = $latestVersion['create_time'] ?? time();
            if (is_numeric($releaseTime)) {
                $releaseTime = date('Y-m-d\TH:i:s', (int)$releaseTime);
            } elseif (is_string($releaseTime)) {
                $timestamp = strtotime($releaseTime);
                $releaseTime = $timestamp ? date('Y-m-d\TH:i:s', $timestamp) : date('Y-m-d\TH:i:s');
            } else {
                $releaseTime = date('Y-m-d\TH:i:s');
            }

            return json([
                'code' => 200,
                'msg'  => '成功',
                'data' => [
                    'id'          => (int)($latestVersion['id'] ?? 0),
                    'userId'      => $userId,
                    'versionCode' => (int)($latestVersion['version'] ?? 0),
                    'forceUpdate' => $forceUpdateRequired,
                    'downloadUrl' => (string)($latestVersion['update_url'] ?? ''),
                    'description' => (string)($latestVersion['description'] ?? ''),
                    'releaseTime' => $releaseTime
                ]
            ], 200, [], JSON_UNESCAPED_SLASHES);
        } catch (\Throwable $e) {
            return json(['code' => 500, 'msg'  => '服务器内部错误', 'data' => null], 200, [], JSON_UNESCAPED_SLASHES);
        }
    }

    public function init()
    {
        $maccms_config = config('maccms') ?? [];
        $app_config = $maccms_config['app'] ?? [];

        try {
            $type_list = Db::name('type')
                ->where(['type_mid' => 1, 'type_status' => 1, 'type_pid' => 0])
                ->order('type_sort asc')
                ->field('type_id, type_name, type_extend')
                ->select() ?: [];
        } catch (\Exception $e) { $type_list = []; }

        $result = [
            'data' => [
                'type_list' => []
            ],
            'msg' => "",
            'code' => 1
        ];

        foreach ($type_list as $type) {
            $type_extends = json_decode($type['type_extend'] ?? '', true);

            $vod_extend_class = isset($type_extends['class']) ? explode(",", $type_extends['class']) : (!empty($app_config['vod_extend_class']) ? explode(",", $app_config['vod_extend_class']) : []);
            $vod_extend_area = isset($type_extends['area']) ? explode(",", $type_extends['area']) : (!empty($app_config['vod_extend_area']) ? explode(",", $app_config['vod_extend_area']) : []);
            $vod_extend_lang = isset($type_extends['lang']) ? explode(",", $type_extends['lang']) : (!empty($app_config['vod_extend_lang']) ? explode(",", $app_config['vod_extend_lang']) : []);
            $vod_extend_year = isset($type_extends['year']) ? explode(",", $type_extends['year']) : (!empty($app_config['vod_extend_year']) ? explode(",", $app_config['vod_extend_year']) : []);
            $vod_extend_sort = ["最新", "最热", "最赞"];

            $filter_type_list = [];

            if (!empty($vod_extend_class) && is_array($vod_extend_class)) {
                $filter_type_list[] = ["name" => "class", "list" => array_merge(["全部"], $vod_extend_class)];
            }
            if (!empty($vod_extend_area) && is_array($vod_extend_area)) {
                $filter_type_list[] = ["name" => "area", "list" => array_merge(["全部"], $vod_extend_area)];
            }
            if (!empty($vod_extend_lang) && is_array($vod_extend_lang)) {
                $filter_type_list[] = ["name" => "lang", "list" => array_merge(["全部"], $vod_extend_lang)];
            }
            if (!empty($vod_extend_year) && is_array($vod_extend_year)) {
                $filter_type_list[] = ["name" => "year", "list" => array_merge(["全部"], $vod_extend_year)];
            }
            $filter_type_list[] = ["name" => "sort", "list" => $vod_extend_sort];

            $result['data']['type_list'][] = [
                "type_id" => $type['type_id'],
                "type_name" => $type['type_name'],
                "filter_type_list" => $filter_type_list
            ];
        }

        return json_encode($result, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
    }


    public function Vodlist()
    {
        $page = max(1, (int)input('page/d', 1));
        $type_id = input('type_id/d', 0);
        $class = input('class/s', '');
        $area = input('area/s', '');
        $lang = input('lang/s', '');
        $year = input('year/s', '');
        $sort = input('sort/s', '');
        $where = ['vod_status' => 1];
        
        if (!empty($class) && $class != "全部") $where['vod_class'] = ["like", "%$class%"];
        if (!empty($area) && $area != "全部") $where['vod_area'] = $area;
        if (!empty($lang) && $lang != "全部") $where['vod_lang'] = $lang;
        if (!empty($year) && $year != "全部") $where['vod_year'] = $year;

        $order = "vod_time desc";
        switch ($sort) {
            case "最新": $order = "vod_time desc"; break;
            case "最热": $order = "vod_hits desc"; break;
            case "最赞": $order = "vod_score desc"; break;
            case "日榜": $order = "vod_hits_day desc"; break;
            case "周榜": $order = "vod_hits_week desc"; break;
            case "月榜": $order = "vod_hits_month desc"; break;
        }

        $type_name = '';

        try {
            $child_type_ids = Db::name('type')
                ->where(['type_mid' => 1, 'type_status' => 1, 'type_pid' => $type_id])
                ->order('type_sort asc')
                ->column('type_id') ?: [];
        } catch (\Exception $e) { $child_type_ids = []; }
        
        $where_type_ids = array_merge([$type_id], $child_type_ids);

        try {
            $recommend_list = Db::name('vod')
                ->where($where)
                ->where(['type_id' => ['in', $where_type_ids]])
                ->order($order)
                ->field($this->simple_vod_fields)
                ->page($page, 30)
                ->select() ?: [];
        } catch (\Exception $e) { $recommend_list = []; }
        
        $recommend_list = $this->replaceVodPic($recommend_list);
        return $this->setData(compact('type_id', 'type_name', 'recommend_list'));
    }

    public function nav()
    {
        try {
            $list = Db::table('appfox_nav')->order('nav_sort', 'DESC')->select() ?: [];
        } catch (\Exception $e) { $list = []; }
        $data = [];
        foreach ($list as $key => $value) {
            $data[$key]['navigationId'] = $value['id'] ?? 0;
            $data[$key]['navigationName'] = $value['name'] ?? '';
        }
        return json(['code' => 200, 'msg' => '成功', 'data' => $data]);
    }
    
    public function nav_video()
    {
        $id = input('id/d', null);
        try {
            if ($id == null) {
                $list = Db::table('appfox_nav')->order('nav_sort', 'DESC')->select() ?: [];
            } else {
                $list = Db::table('appfox_nav')->where('id', $id)->select() ?: [];
            }
        } catch (\Exception $e) { $list = []; }

        $data = [];
        foreach ($list as $key => $value) {
            $navId = $value['id'] ?? 0;
            $data[$key]['navigationId'] = $navId;
            $data[$key]['navigationName'] = $value['name'] ?? '';

            try {
                $categories = Db::table('appfox_module')
                    ->where('nav_id', $navId)
                    ->order('sort_order', 'DESC')
                    ->select() ?: [];
            } catch (\Exception $e) { $categories = []; }

            $data[$key]['categories'] = [];
            foreach ($categories as $k => $v) {
                $modType = $v['module_type'] ?? 1;
                $catId = $v['category_id'] ?? 0;
                $num = $v['num'] ?? 10;
                $bindRec = $v['bind_recommend'] ?? 0;
                
                $data[$key]['categories'][$k]['categoriesType'] = $modType;
                $data[$key]['categories'][$k]['module_id'] = $v['id'] ?? 0;
                $data[$key]['categories'][$k]['name'] = $v['module_name'] ?? '';
                $data[$key]['categories'][$k]['videos'] = [];

                if ($modType == 3) {
                    $data[$key]['categories'][$k]['videosDate'] = [];
                    $weekdaylist = ['日', '一', '二', '三', '四', '五', '六'];

                    try {
                        $video_query = Db::name('vod')->field('vod_id,vod_name,vod_sub,vod_pic,vod_remarks,vod_actor,vod_blurb,vod_pic_slide,vod_pubdate,vod_time,vod_weekday')->order('vod_time', 'DESC');
                        if ($catId != 999) $video_query->where('type_id', $catId);
                        $video_list = $video_query->select() ?: [];
                    } catch (\Exception $e) { $video_list = []; }

                    $grouped = [];
                    foreach ($video_list as $row) {
                        $wStr = $row['vod_weekday'] ?? '';
                        foreach ($weekdaylist as $w) {
                            if (strpos($wStr, $w) !== false) $grouped[$w][] = $row;
                        }
                    }

                    foreach ($weekdaylist as $kk => $vv) {
                        $videos = isset($grouped[$vv]) ? array_slice($grouped[$vv], 0, $num) : [];
                        $data[$key]['categories'][$k]['videosDate'][$kk]['week'] = $vv;
                        $data[$key]['categories'][$k]['videosDate'][$kk]['videos'] = [];

                        foreach ($videos as $kks => $vvs) {
                            $data[$key]['categories'][$k]['videosDate'][$kk]['videos'][$kks] = [
                                'vod_id' => $vvs['vod_id'] ?? 0, 'vod_name' => $vvs['vod_name'] ?? '',
                                'vod_sub' => $vvs['vod_sub'] ?? '', 'vod_pic' => $vvs['vod_pic'] ?? '',
                                'vod_remarks' => $vvs['vod_remarks'] ?? '', 'vod_actor' => $vvs['vod_actor'] ?? '',
                                'vod_blurb' => $vvs['vod_blurb'] ?? '', 'vod_pic_slide' => $vvs['vod_pic_slide'] ?? '',
                                'vod_pubdate' => $vvs['vod_pubdate'] ?? '', 'vod_time' => $vvs['vod_time'] ?? 0,
                            ];
                        }
                    }
                } else {
                    try {
                        $video_query = Db::name('vod')->field('vod_id,vod_name,vod_sub,vod_pic,vod_remarks,vod_actor,vod_blurb,vod_pic_slide,vod_pubdate,vod_time,vod_hits_day,type_id')->limit($num);
                        if ($catId && $catId != 999) $video_query->where('type_id|type_id_1', 'eq', $catId);

                        if ($bindRec >= 51 && $bindRec <= 54) {
                            $type_names = ['vod_hits_day', 'vod_hits_week', 'vod_hits_month', 'vod_hits'];
                            $video_query->order($type_names[$bindRec - 51], 'DESC');
                        } else {
                            $video_query->where('vod_level', $bindRec)->order('vod_time', 'DESC');
                        }
                        $video_list = $video_query->select() ?: [];
                    } catch (\Exception $e) { $video_list = []; }

                    if ($modType == 2 && is_array($video_list)) {
                        usort($video_list, function ($a, $b) {
                            $timeA = strtotime($a['vod_pubdate'] ?? ''); $timeB = strtotime($b['vod_pubdate'] ?? '');
                            if ($timeA === false && $timeB === false) return 0;
                            if ($timeA === false) return 1;
                            if ($timeB === false) return -1;
                            return $timeB <=> $timeA;
                        });
                        $video_list = array_slice($video_list, 0, $num);
                    }

                    foreach ($video_list as $kk => $vv) {
                        if ($bindRec >= 51 && $bindRec <= 54) {
                            $type_names = ['日人气', '周人气', '月人气', '总人气'];
                            $data[$key]['categories'][$k]['categoriesName'] = $type_names[$bindRec - 51];
                        } else {
                            $data[$key]['categories'][$k]['categoriesName'] = '推荐'.$bindRec;
                        }
                        $data[$key]['categories'][$k]['videos'][$kk] = [
                            'vod_id' => $vv['vod_id'] ?? 0, 'type_id'=>$vv['type_id'] ?? 0,
                            'vod_name' => $vv['vod_name'] ?? '', 'vod_sub' => $vv['vod_sub'] ?? '',
                            'vod_pic' => $vv['vod_pic'] ?? '', 'vod_remarks' => $vv['vod_remarks'] ?? '',
                            'vod_actor' => $vv['vod_actor'] ?? '', 'vod_blurb' => $vv['vod_blurb'] ?? '',
                            'vod_pic_slide' => $vv['vod_pic_slide'] ?? '', 'vod_pubdate' => $vv['vod_pubdate'] ?? '',
                            'vod_time' => $vv['vod_time'] ?? 0, 'vod_hits_day'=>$vv['vod_hits_day'] ?? 0
                        ];
                    }
                }
            }
        }
        return json(['code' => 200, 'msg' => '成功', 'data' => $data]);
    }

    public function module_detail()
    {
        $id = input('id/d', null);
        $page = max(1, (int)input('page/d', 1));
        
        try {
            $categories = Db::table('appfox_module')->where('id', $id)->find();
        } catch (\Exception $e) { return json(['code' => 404, 'msg' => '模块不存在']); }

        if (empty($categories)) return json(['code' => 404, 'msg' => '模块不存在']);

        $data = [];
        $data['categoriesType'] = $categories['module_type'] ?? 1;
        $data['name'] = $categories['module_name'] ?? '';

        $perPage = 15;
        $catId = $categories['category_id'] ?? 0;
        $bindRec = $categories['bind_recommend'] ?? 0;

        if ($data['categoriesType'] == 3) {
            $data['videosDate'] = [];
            $weekdaylist = ['日', '一', '二', '三', '四', '五', '六'];
            foreach ($weekdaylist as $kk => $vv) {
                try {
                    $query = Db::name('vod')->where('vod_weekday', 'like', '%' . $vv . '%')->order('vod_time', 'DESC')->page($page, $perPage);
                    if ($catId != 999) $query->where('type_id', $catId);
                    $video_list = $query->select() ?: [];
                } catch (\Exception $e) { $video_list = []; }

                $data['videosDate'][$kk]['week'] = $vv;
                $data['videosDate'][$kk]['videos'] = [];

                foreach ($video_list as $kks => $vvs) {
                    $data['videosDate'][$kk]['videos'][$kks] = [
                        'vod_id' => $vvs['vod_id'] ?? 0, 'vod_name' => $vvs['vod_name'] ?? '',
                        'vod_sub' => $vvs['vod_sub'] ?? '', 'vod_pic' => $vvs['vod_pic'] ?? '',
                        'vod_remarks' => $vvs['vod_remarks'] ?? '', 'vod_actor' => $vvs['vod_actor'] ?? '',
                        'vod_blurb' => $vvs['vod_blurb'] ?? '', 'vod_pic_slide' => $vvs['vod_pic_slide'] ?? '',
                        'vod_pubdate' => $vvs['vod_pubdate'] ?? '', 'vod_time' => $vvs['vod_time'] ?? 0,
                    ];
                }
            }
        } else {
            $data['videos'] = [];
            try {
                $query = Db::name('vod')->where('vod_level', $bindRec)->order('vod_time', 'DESC');
                if ($catId != 999) $query->where('type_id', $catId);
                $video_lists = $query->paginate(['list_rows' => $perPage, 'page' => $page]);
                $count = $video_lists->total();
            } catch (\Exception $e) {
                $video_lists = []; $count = 0;
            }

            foreach ($video_lists as $kk => $vv) {
                $data['videos'][$kk] = [
                    'vod_id' => $vv['vod_id'] ?? 0, 'vod_name' => $vv['vod_name'] ?? '',
                    'vod_sub' => $vv['vod_sub'] ?? '', 'vod_pic' => $vv['vod_pic'] ?? '',
                    'vod_remarks' => $vv['vod_remarks'] ?? '', 'vod_actor' => $vv['vod_actor'] ?? '',
                    'vod_blurb' => $vv['vod_blurb'] ?? '', 'vod_pic_slide' => $vv['vod_pic_slide'] ?? '',
                    'vod_pubdate' => $vv['vod_pubdate'] ?? '', 'vod_time' => $vv['vod_time'] ?? 0,
                ];
            }
        }

        return json([
            'code' => 200, 'msg' => '成功', 'page' => $page,
            'count' => $count ?? 0, 'page_count' => isset($count) ? ceil($count / $perPage) : 0,
            'data' => $data
        ]);
    }

    public function top()
    {
        try {
            $all_type_list = Db::name('Type')->where('type_status', 1)->field('*')->order('type_id ASC')->select() ?: [];
            $top_order_row = Db::table('appfox_system_config')->find();
        } catch (\Exception $e) { $all_type_list = []; $top_order_row = []; }
        
        $top_order = !empty($top_order_row['top_order']) ? $top_order_row['top_order'] : 'vod_id';
        $data = [];
        
        foreach ($all_type_list as $key => $value) {
            $data[$key]['navigationId'] = $value['type_id'] ?? 0;
            $data[$key]['navigationName'] = $value['type_name'] ?? '';
            $data[$key]['videos'] = [];
            
            try {
                $videos = Db::name('vod')->where('type_id|type_id_1', 'eq', $value['type_id'])->order($top_order . ' desc')->limit(20)->select() ?: [];
            } catch (\Exception $e) { $videos = []; }
            
            foreach ($videos as $k => $v) {
                $data[$key]['videos'][$k] = [
                    'vod_id' => $v['vod_id'] ?? 0, 'vod_name' => $v['vod_name'] ?? '',
                    'vod_hits' => $v['vod_hits'] ?? 0, 'vod_sub' => $v['vod_sub'] ?? '',
                    'vod_pic' => $v['vod_pic'] ?? '', 'vod_remarks' => $v['vod_remarks'] ?? '',
                    'vod_actor' => $v['vod_actor'] ?? '', 'vod_blurb' => $v['vod_blurb'] ?? '',
                    'vod_pic_slide' => $v['vod_pic_slide'] ?? '', 'vod_pubdate' => $v['vod_pubdate'] ?? '',
                    'vod_time' => $v['vod_time'] ?? 0
                ];
            }
        }
        return json(['code' => 200, 'msg' => '成功', 'data' => $data]);
    }

    public function commentList()
    {
        $vod_id = input('vod_id/d', 1);
        $page = max(1, (int)input('page/d', 1));
        $sort = input('sort/d', 1);
        $comment_list = $this->getCommentList($vod_id, $page, $sort);

        return json(['code' => 200, 'msg' => '成功', 'data' => $comment_list]);
    }


    private function getCommentList($vod_id, $page, $sort = 1)
    {
        $order = ($sort == 1) ? 'comment_id desc' : 'comment_reply desc';

        try {
            $comment_list = Db::name('comment')
                ->where(['comment_pid' => 0, 'comment_mid' => 1, 'comment_status' => 1, 'comment_rid' => $vod_id])
                ->order($order)->page($page, 10)->select() ?: [];
        } catch (\Exception $e) { $comment_list = []; }

        $config = config('maccms') ?? [];
        $filter_words = explode(",", $config['app']['filter_words'] ?? '');
        
        if (!empty($comment_list)) {
            $comment_list_ids = array_column($comment_list, 'comment_id');
            try {
                $child_comment_list = Db::name('comment')
                    ->whereIn('comment_pid', $comment_list_ids)->order('comment_id asc')->column('*', 'comment_pid') ?: [];
            } catch (\Exception $e) { $child_comment_list = []; }
            
            foreach ($comment_list as $key => $comment) {
                if (isset($child_comment_list[$comment['comment_id']])) {
                    $child_comment_content = $child_comment_list[$comment['comment_id']]['comment_content'] ?? '';
                    $child_comment_content = $this->filterWords($child_comment_content, $filter_words);
                    $comment_list[$key]['child_comment_content'] = $child_comment_content;
                    $comment_list[$key]['child_user_id'] = $child_comment_list[$comment['comment_id']]['user_id'] ?? 0;
                } else {
                    $comment_list[$key]['child_comment_content'] = '';
                    $comment_list[$key]['child_user_id'] = 0;
                }
            }

            $comment_user_ids = array_column($comment_list, 'user_id');
            $child_user_id = array_column($comment_list, 'child_user_id');
            $all_user_ids = array_unique(array_merge($comment_user_ids, $child_user_id));

            try {
                $users = Db::name('user')->whereIn('user_id', $all_user_ids)->column('*', 'user_id') ?: [];
            } catch (\Exception $e) { $users = []; }

            $users[0] = ['user_nick_name' => '游客', 'user_name' => '游客'];
            $users_extra = [];
            $users_extra[0] = ['avatar_update_time' => 0];

            foreach ($comment_list as $key => &$comment) {
                $uid = $comment['user_id'] ?? 0;
                $cuid = $comment['child_user_id'] ?? 0;
                
                $user = $users[$uid] ?? $users[0];
                $user_extra = $users_extra[$uid] ?? $users_extra[0];
                $child_user = $users[$cuid] ?? $users[0];

                $comment_name = empty($user['user_nick_name']) ? ($user['user_name'] ?? '游客') : $user['user_nick_name'];
                $child_user_name = empty($child_user['user_nick_name']) ? ($child_user['user_name'] ?? '游客') : $child_user['user_nick_name'];

                $comment['comment_content'] = $this->filterWords($comment['comment_content'] ?? '', $filter_words);
                $comment['comment'] = $comment['comment_content'];

                $comment['user_name'] = $this->maskPhoneNumber($comment_name);
                $comment['child_user_name'] = $this->maskPhoneNumber($child_user_name);
                
                $comment['user_avatar'] = $this->getUserAvatar($user);
                $comment['create_time'] = date("Y-m-d H:i", (int)($comment['comment_time'] ?? time()));

                $parse_arr = $this->parseTimeByText($comment['comment']);
                $comment['time_str'] = $parse_arr['time_str'];
                $comment['seek_to_time'] = $parse_arr['seek_to_time'];
            }
        }
        return $comment_list;
    }

    public function sendComment()
    {
        $config = config('maccms') ?? [];
        $comment_config = $config['comment'] ?? ['status' => 0, 'audit' => 1];

        $comment_text = input('comment/s', '');
        $vod_id = input('vod_id/d', 0);
        $user_id = input('user_id/d', 0);
        $user_name = input('user_name/s', '');
        $reply_comment_id = input('comment_id/d', 0);

        if (!$comment_config['status']) return $this->setMsg('评论未开启');
        if (empty($comment_text)) return $this->setMsg('评论不能为空');
        if (mb_strlen($comment_text) > 200) return $this->setMsg("评论文字过长");
        if ($user_id <= 0 || empty($user_name)) return $this->setMsg("用户信息错误，请重新登录");

        $comment_status = ($comment_config['audit'] == 1) ? 0 : 1;

        $comment = [
            'comment_mid' => 1, 'comment_rid' => intval($vod_id), 'user_id' => $user_id,
            'comment_time' => time(), 'comment_content' => $comment_text, 'comment_name' => $user_name,
            'comment_status' => $comment_status, 'comment_pid' => $reply_comment_id,
            'comment_ip' => $this->mac_get_ip_long()
        ];

        try {
            $comment_id = Db::name('comment')->insertGetId($comment);
            $is_reply = false;
            if ($reply_comment_id > 0) {
                $is_reply = true;
                Db::name('comment')->where(['comment_id' => $reply_comment_id])->setInc('comment_reply');
            }
        } catch (\Exception $e) { return $this->setMsg("系统异常，评论失败"); }

        $parse_arr = $this->parseTimeByText($comment_text);
        $comment['comment_id'] = $comment_id;
        $comment['create_time'] = date("Y-m-d H:i");
        $comment['user_name'] = $user_name; 
        $comment['user_avatar'] = ''; 
        $comment['comment'] = $comment_text;
        $comment['is_reply'] = $is_reply;
        $comment['user_is_vip'] = 0; 
        $comment['time_str'] = $parse_arr['time_str'];
        $comment['seek_to_time'] = $parse_arr['seek_to_time'];

        return json(['code' => 200, 'msg' => $comment_status ? "评论发送成功！" : "评论审核中！", 'data' => ['status' => !empty($comment_status), 'comment' => $comment]]);
    }

    public function childrenCommentList()
    {
        $reply_comment_id = input('comment_id/d', 0);
        $page = max(1, (int)input('page/d', 1));
        
        try {
            $parent_comment = Db::name('comment')->where(['comment_id' => $reply_comment_id])->find();
        } catch (\Exception $e) { return $this->setMsg('评论不存在'); }
        
        if (empty($parent_comment)) return $this->setMsg('评论不存在');
        
        try {
            $children_comment_list = Db::name('comment')->where(['comment_mid' => 1, 'comment_pid' => $reply_comment_id, 'comment_status' => 1])->order('comment_id desc')->page($page, 10)->select() ?: [];
        } catch (\Exception $e) { $children_comment_list = []; }

        if ($page == 1) array_unshift($children_comment_list, $parent_comment);

        if (!empty($children_comment_list)) {
            $comment_user_ids = array_column($children_comment_list, 'user_id');
            try {
                $users = Db::name('user')->whereIn('user_id', $comment_user_ids)->column('*', 'user_id') ?: [];
            } catch (\Exception $e) { $users = []; }

            $users[0] = ['user_nick_name' => '游客', 'user_name' => '游客'];
            $users_extra = [];
            $users_extra[0] = ['avatar_update_time' => 0];

            foreach ($children_comment_list as $key => &$comment) {
                $uid = $comment['user_id'] ?? 0;
                $cuid = $comment['child_user_id'] ?? 0;
                
                $user = $users[$uid] ?? $users[0];
                $user_extra = $users_extra[$uid] ?? $users_extra[0];
                $child_user = $users[$cuid] ?? $users[0];

                $comment_name = empty($user['user_nick_name']) ? ($user['user_name'] ?? '游客') : $user['user_nick_name'];
                $child_user_name = empty($child_user['user_nick_name']) ? ($child_user['user_name'] ?? '游客') : $child_user['user_nick_name'];

                $comment['comment'] = $comment['comment_content'] ?? '';
                $comment['user_name'] = $this->maskPhoneNumber($comment_name);
                $user['avatar_update_time'] = $user_extra['avatar_update_time'] ?? 0;
                $comment['user_avatar'] = $this->getUserAvatar($user);
                $comment['create_time'] = date("Y-m-d H:i", (int)($comment['comment_time'] ?? time()));
                $comment['child_user_name'] = $this->maskPhoneNumber($child_user_name);
                
                $parse_arr = $this->parseTimeByText($comment['comment']);
                $comment['time_str'] = $parse_arr['time_str'];
                $comment['seek_to_time'] = $parse_arr['seek_to_time'];
            }
        }
        return $this->setData(compact('children_comment_list'));
    }

    protected function macFilterXss($str)
    {
        return trim(htmlspecialchars(strip_tags($str), ENT_QUOTES));
    }

    protected function setMsg($msg = '', $code = 0, $data = [])
    {
        if (!is_array($data)) $data = [];
        return json(['code' => $code, 'msg' => $msg, 'data' => $data]);
    }

    public function register()
    {
        $username = request()->param('username');
        $password = request()->param('password');
        $nickname = request()->param('nickname'); 

        if (empty($username) || empty($password)) return json(['code' => 400, 'msg' => '用户名或密码不能为空']);

        try {
            $exists = Db::name('user')->where('user_name', $username)->find();
            if ($exists) return json(['code' => 400, 'msg' => '用户名已存在']);

            $random = substr(md5(uniqid(mt_rand(), true)), 0, 10);
            $pwd = md5($password . $random);

            $data = [
                'user_name' => $username, 'user_pwd' => $pwd, 'user_random' => $random,
                'user_reg_time' => time(), 'user_status' => 1, 'user_login_time' => time(),
                'user_nick_name' => $nickname ?: $username 
            ];
            $userId = Db::name('user')->insertGetId($data);

            return json(['code' => 200, 'msg' => '注册成功', 'data' => ['user_id' => $userId, 'username' => $username, 'nickname' => $data['user_nick_name'], 'token' => md5($userId . $pwd . time())]]);
        } catch (\Exception $e) { return json(['code' => 500, 'msg' => '注册失败']); }
    }


    protected function getUserAvatar($user_info)
    {
        $avatar = "";
        if (!empty($user_info['user_portrait'])) {
            $config = config('maccms') ?? [];
            $upload_config = $config['upload'] ?? [];
            $protocol = $upload_config['protocol'] ?? 'http';
            $user_info['user_portrait'] = str_replace("mac://", "{$protocol}://", $user_info['user_portrait']);
            $res = $user_info['user_portrait'];
            if (stristr($res, "http:") || stristr($res, "https:")) return $res;
        } else {
            $user_id = $user_info['user_id'] ?? 0;
            $res = 'upload/user/' . ($user_id % 10) . '/' . $user_id . '.jpg';
        }
        if (file_exists(ROOT_PATH . $res)) {
            $t = $user_info['user_end_time'] ?? time();
            $avatar = $this->getImgUrl($res . "?t=" . $t);
        }
        return $avatar;
    }

    public function login()
    {
        $username = request()->param('username');
        $password = request()->param('password');

        try {
            $user = Db::name('user')->where('user_name', $username)->find();
            if (!$user) return json(['code' => 400, 'msg' => '用户不存在']);

            $pwdCheck = md5($password . ($user['user_random'] ?? ''));
            if ($pwdCheck !== ($user['user_pwd'] ?? '')) return json(['code' => 400, 'msg' => '密码错误']);
            if (($user['user_status'] ?? 0) != 1) return json(['code' => 403, 'msg' => '账号被禁用']);

            $token = md5(uniqid() . $user['user_id'] . time());

            Db::name('user')->where('user_id', $user['user_id'])->update([
                'user_token' => $token, 'user_login_time' => time(), 'user_login_num' => Db::raw('user_login_num+1'),
            ]);

            return json(['code' => 200, 'msg' => '登录成功', 'data' => ['user_id' => $user['user_id'], 'username' => $user['user_name'], 'nickname' => $user['user_nick_name'] ?? '', 'email' => $user['user_email'] ?? '', 'phone' => $user['user_phone'] ?? '', 'token' => $token, 'avatar' => $this->getUserAvatar($user) ?? '', 'last_login' => time()]]);
        } catch (\Exception $e) { return json(['code' => 500, 'msg' => '登录异常']); }
    }

    public function uploadAvatarByUserId()
    {
        $userId = request()->param('user_id/d', 0);
        if (empty($userId)) return json(['code' => 400, 'msg' => '缺少 user_id']);

        try {
            $user = Db::name('user')->where('user_id', $userId)->find();
            if (!$user) return json(['code' => 404, 'msg' => '用户不存在']);
            if (($GLOBALS['config']['user']['portrait_status'] ?? 0) == 0) return json(['code' => 403, 'msg' => lang('index/portrait_tip1')]);

            $data = ['flag' => 'user', 'user_id' => $userId, 'input' => 'file'];
            $res = model('Upload')->upload($data);
            if (!$res['code']) return json(['code' => 400, 'msg' => $res['msg']]);

            $user_portrait = $res['data']['file'] ?? '';
            if (empty($user_portrait)) return json(['code' => 400, 'msg' => '上传失败']);

            Db::name('user')->where('user_id', $userId)->update(['user_portrait' => $user_portrait, 'user_end_time' => time()]);
            return json(['code' => 200, 'msg' => '修改成功', 'data' => ['user_id' => $userId, 'avatar' => $this->getUserAvatar($user)]]);
        } catch (\Exception $e) { return json(['code' => 500, 'msg' => '上传异常']); }
    }


    protected function maskPhoneNumber($user_name)
    {
        if (preg_match('/^1\d{10}$/', $user_name)) return substr($user_name, 0, 3) . '****' . substr($user_name, -4);
        return $user_name;
    }

    public function updateNickname()
    {
        $user_id = request()->param('user_id/d', 0);
        $nickname = request()->param('nickname/s', '');

        if (empty($user_id) || empty($nickname)) return json(['code' => 400, 'msg' => '缺少参数']);
        if (mb_strlen($nickname) < 2 || mb_strlen($nickname) > 20) return json(['code' => 400, 'msg' => '昵称长度需为2-20个字符']);

        try {
            $user = Db::name('user')->where('user_id', $user_id)->find();
            if (!$user) return json(['code' => 404, 'msg' => '用户不存在']);

            $exists = Db::name('user')->where('user_nick_name', $nickname)->where('user_id', '<>', $user_id)->find();
            if ($exists) return json(['code' => 409, 'msg' => '昵称已被占用']);

            $res = Db::name('user')->where('user_id', $user_id)->update(['user_nick_name' => $nickname]);
            if ($res !== false) return json(['code' => 200, 'msg' => '昵称修改成功', 'data' => ['nickname' => $nickname]]);
            return json(['code' => 500, 'msg' => '修改失败，请稍后重试']);
        } catch (\Exception $e) { return json(['code' => 500, 'msg' => '系统异常']); }
    }

    protected function filterWords($words, $filter_words = [], $type = 'app')
    {
        if (empty($filter_words)) {
            $config = config('maccms') ?? [];
            $filter_words = explode(",", $config[$type]['filter_words'] ?? '');
        }
        return str_replace($filter_words, "***", (string)$words);
    }

    protected function mac_get_client_ip()
    {
        static $final;
        if (!is_null($final)) return $final;
        $ips = [];
        if (!empty($_SERVER['HTTP_CF_CONNECTING_IP'])) $ips[] = $_SERVER['HTTP_CF_CONNECTING_IP'];
        if (!empty($_SERVER['HTTP_ALI_CDN_REAL_IP'])) $ips[] = $_SERVER['HTTP_ALI_CDN_REAL_IP'];
        if (!empty($_SERVER['HTTP_CLIENT_IP'])) $ips[] = $_SERVER['HTTP_CLIENT_IP'];
        if (!empty($_SERVER['HTTP_PROXY_USER'])) $ips[] = $_SERVER['HTTP_PROXY_USER'];
        $real_ip = getenv('HTTP_X_REAL_IP');
        if (!empty($real_ip)) $ips[] = $real_ip;
        if (!empty($_SERVER['REMOTE_ADDR'])) $ips[] = $_SERVER['REMOTE_ADDR'];
        foreach ($ips as $ip) {
            $verifyResult = filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_RES_RANGE);
            if (!$verifyResult) continue;
            $verifyResult && $final = $ip;
        }
        empty($final) && $final = '0.0.0.0';
        return $final;
    }

    protected function mac_get_ip_long($ip_addr = '')
    {
        $ip_addr = !empty($ip_addr) ? $ip_addr : $this->mac_get_client_ip();
        $ip_long = sprintf('%u', ip2long($ip_addr));
        if ($ip_long < 0 || $ip_long >= 0xFFFFFFFF) $ip_long = 0;
        return $ip_long;
    }

    protected function parseTimeByText($text)
    {
        $pattern = '/(?:(\d{1,2})[:：])?(\d{1,2})[:：](\d{1,2})/u';
        $result_array = ['time_str' => '', 'seek_to_time' => 0];

        if (preg_match_all($pattern, $text, $matches)) {
            foreach ($matches[0] as $match) {
                $timeParts = explode(':', str_replace("：", ":", $match));
                $seconds = 0;
                if (count($timeParts) === 3) {
                    $seconds = ($timeParts[0] * 3600) + ($timeParts[1] * 60) + $timeParts[2]; 
                } elseif (count($timeParts) === 2) {
                    $seconds = ($timeParts[0] * 60) + $timeParts[1]; 
                } elseif (count($timeParts) === 1) {
                    $seconds = $timeParts[0]; 
                }
                $result_array = ['time_str' => $match, 'seek_to_time' => $seconds * 1000];
            }
        }
        return $result_array; 
    }
}